#version 150

/**
 * © Janus Bager Kristensen & Rolf Bagge, CAVI Aarhus University, 2014
 */

uniform sampler2D extractedCenters;
uniform sampler2D originalGrayscale;

uniform float radius;
uniform float fingerRadius;
uniform float ringSize;

const int circleSteps = 256;
const int circleStepsPrecise = 1080; // How many steps around a FULL circle
const float pi = 3.1415926535;
const float twopi = 3.1415926535 * 2.0;
const float piStep = (2.0 * pi) / circleSteps;
const float precisePiStep = (2.0 * pi) / circleStepsPrecise;

//360 degrees on circle, so 36 steps is 10 degrees
const float tenDegrees = (2 * pi) / 36;

smooth in vec2 texCoord;

out vec4 outColor;

vec2 radiansToUV(float radians, vec2 circleCenter, float radius) {
    vec2 textureUnits = 1.0 / textureSize(originalGrayscale, 0);

    vec2 circleUV;
    circleUV.x = circleCenter.x + radius * cos(radians) * textureUnits.x;
    circleUV.y = circleCenter.y + radius * sin(radians) * textureUnits.y;

    return circleUV;
}

float rotationValue(float radians, vec2 circleCenter, float radius) {
        float knobRadians = radians;
        float oppositeKnobRadians = radians + pi;
        float oppositeWhiteKnobRadians = oppositeKnobRadians + tenDegrees;
        float oppositeWhiteKnobRadians2 = oppositeKnobRadians - tenDegrees;

        float knobBlackRadians = knobRadians + 2.0*tenDegrees;
        float oppositeKnobBlackRadians = oppositeWhiteKnobRadians + 2.0*tenDegrees;

        float knob = texture(originalGrayscale, radiansToUV(knobRadians, circleCenter, radius)).r; // should be white
        float oppositeWhite = texture(originalGrayscale, radiansToUV(oppositeWhiteKnobRadians, circleCenter, radius)).r; // should be black
        float oppositeWhite2 = texture(originalGrayscale, radiansToUV(oppositeWhiteKnobRadians2, circleCenter, radius)).r; // should be black
        float knobBlack = texture(originalGrayscale, radiansToUV(knobBlackRadians, circleCenter, radius)).r;
        float oppositeWhiteKnobBlack = texture(originalGrayscale, radiansToUV(oppositeKnobBlackRadians, circleCenter, radius)).r;

        return min(knob - knobBlack, (oppositeWhite+oppositeWhite2)/2.0 - oppositeWhiteKnobBlack);
}

vec2 weightedRotationValue(float radians, vec2 circleCenter, float radius) {

        float value = rotationValue(radians, circleCenter, radius);

        vec2 result;
        result.x = radians * value;
        result.y = value;

        return result;
}

float findRotationKnobRadians(vec2 circleCenter, float radius) {

    float currentValue = 0;
    float maxValue = 0;
    float maxRadians = 0;

    //for(float r = radius-2; r<=radius+2; r++) {
    float r = radius;
        for(int i = 0; i<circleSteps; i++) {
            float radians = i * piStep;

            currentValue = rotationValue(radians, circleCenter, r);

            if(currentValue > maxValue) {
                maxValue = currentValue;
                maxRadians = radians;
            }
        }
    //}
    
    return maxRadians;
}

vec2 findOptimalRadians(float radians, vec2 circleCenter, float radius, float ringSize) {
    float startRadians = radians - tenDegrees;
    float endRadians = radians + tenDegrees;

    //X: radians * value
    //Y: totalValue
    vec2 result = vec2(0);

    for(float i = -1; i<=1; i++) {
        float sampleRadius = radius + i*(ringSize/4.0);
        for(float r = startRadians; r < endRadians; r+= precisePiStep) {
            result += weightedRotationValue(r, circleCenter, sampleRadius);
        }
    }

    //Average over the totalValue number
    result.x = result.x / result.y;

    return result;
}

void main()
{
    vec4 center = texture(extractedCenters, texCoord);

    outColor = vec4(0);
    vec2 circleCenterUV = center.xy;

    if(center.a > 0.0) {
        // For tags make a more involved search for higher precision
        float outerRadius = radius + ringSize / 2.0;
        float radians = findRotationKnobRadians(circleCenterUV, outerRadius);
        vec2 result = findOptimalRadians(radians, circleCenterUV, outerRadius, ringSize);

        result.x = twopi - result.x;

        if(result.x < 0) {
            result.x += twopi;
        }
        if(result.x > twopi) {
            result.x -= twopi;
        }

        outColor = vec4(result, 1.0, 1.0);
    }
}
