#version 150

/**
 * © Janus Bager Kristensen & Rolf Bagge, CAVI Aarhus University, 2014
 */

uniform sampler2D mask;
uniform sampler2D inputTex;

smooth in vec2 texCoord;

out vec4 fragOut;

vec3 HSVToRGB(float H, float S, float V) {
    float C = V * S;

    float H2 = H / (60.0/360.0);

    float X = C * (1.0 - abs(mod(H2,2)-1.0));

    vec3 rgb = vec3(1);

    if(0 <= H2 && H2 < 1) {
        rgb = vec3(C,X,0);
    } else if(H2 < 2) {
        rgb = vec3(X,C,0);
    } else if(H2 < 3) {
        rgb = vec3(0,C,X);
    } else if(H2 < 4) {
        rgb = vec3(0,X,C);
    } else if(H2 < 5) {
        rgb = vec3(X,0,C);
    } else if(H2 < 6) {
        rgb = vec3(C,0,X);
    }

    float m = V - C;

    rgb += m;

    return rgb;
}

vec3 RGBToHSV(float R, float G, float B) {
    float M = max(max(R,G),B);

    float m = min(min(R,G),B);

    float C = M - m;

    float H = 0;
    float S = 0;
    float V = 0;
    
    float H2 = 0;

    if(C == 0) {
        H2 = 0;
    } else if(M == R) {
        H2 = mod((G-B) / C, 6.0);
    } else if(M == G) {
        H2 = (B-R) / C + 2.0;
    } else if(M == B) {
        H2 = (R-G) / C + 4.0;
    }

    H = (60 * H2) / 360.0;
    V = M;
    
    if(C > 0) {
        S = C/V;
    } else {
        S = 0;
    }

    return vec3(H,S,V);
}

void main() {

    //Get rgb value from input texture
    vec3 rgb = texture(inputTex, texCoord).rgb;

    //Convert rgb to hsv
    vec3 hsv = RGBToHSV(rgb.r, rgb.g, rgb.b);

    //Texture coords for mask lookup based on HSV
    vec2 maskUV = vec2(hsv.x, 1.0-hsv.y);

    //Get mask value from mask texture
    vec4 mask = texture(mask, maskUV);

    //Min V is in red
    float minV = mask.r;
    
    //Max V is in green
    float maxV = mask.g;

    //Falloff V fraction of half the length (maxV-minV)
    float falloffV = mask.b;

    //Init fraction with HS fraction
    float fraction = mask.a;

    if(hsv.z >= minV && hsv.z <= maxV) {
    } else {
        //Calculate half the length of the V interval
        float halfLength = (maxV - minV) / 2.0;

        //Calculate the value of the middle point in the V interval
        float middleV = minV + halfLength;

        //Caluclate the falloff start
        float falloffDistance = falloffV;

        //Calculate the distance from the texture V to the middle of V interval
        float distanceToMiddleV = abs(middleV - hsv.z);

        if((distanceToMiddleV-halfLength) < falloffDistance) {
            //Check if we are at the falloff section of the interval
            //Calculate how far into the falloff section we are
            float falloffFraction = 1.0 - clamp((distanceToMiddleV - halfLength), 0, falloffDistance) / falloffDistance;

            //Update fraction with V falloff
            fraction *= falloffFraction;
        } else {
            fraction = 0;
        }
    }

    fragOut = vec4(vec3(1.0 - fraction),1);
}